from flask import Blueprint, render_template, redirect, url_for, flash, request, make_response
from app.forms import RequestQuoteForm, ContactForm, BookingForm
from app.models import Car, Tour, Booking, VehicleCategory, TourType, Contact, Blog, TourParentCategory, TourSubCategory, Blog
from flask_mail import Message
from flask import current_app
from .models import QuoteRequest
from urllib.parse import urljoin
from sqlalchemy.orm import joinedload
from datetime import datetime, timedelta
import xml.etree.ElementTree as ET
from xml.dom import minidom
import logging
from app import mail, db
from sqlalchemy import desc

main = Blueprint('main', __name__)

def get_popular_tours():
    return Tour.query.order_by(Tour.views.desc()).all()

@main.context_processor
def inject_popular_tours():
    return dict(popular_tours=get_popular_tours())

@main.route('/sitemap.xml')
def sitemap():
    try:
        """Generate sitemap.xml. Makes a list of urls and date modified."""
        pages = []
        ten_days_ago = datetime.now() - timedelta(days=10)
        
        # Static routes
        for rule in current_app.url_map.iter_rules():
            if "GET" in rule.methods and len(rule.arguments) == 0:
                # Exclude admin routes and other unwanted routes
                if not rule.rule.startswith('/admin') and not rule.rule.startswith('/static'):
                    pages.append([url_for(rule.endpoint, _external=True), ten_days_ago])

        # Dynamic routes
        tours = Tour.query.all()
        blogs = Blog.query.all()
        tour_categories = TourParentCategory.query.all()

        # Car rentals page
        pages.append([url_for('main.car_rentals', _external=True), ten_days_ago])

        # Tour category pages
        for category in tour_categories:
            url = url_for('main.tour_packages', category_slug=category.slug, _external=True)
            pages.append([url, ten_days_ago])

        # Tour pages
        for tour in tours:
            url = url_for('main.tour_detail', slug=tour.slug, _external=True)
            pages.append([url, ten_days_ago])

        # Blog pages
        for blog in blogs:
            url = url_for('main.blog_detail', slug=blog.slug, _external=True)
            pages.append([url, ten_days_ago])

        # Generate XML sitemap
        urlset = ET.Element("urlset")
        urlset.set("xmlns", "http://www.sitemaps.org/schemas/sitemap/0.9")

        for url, date in pages:
            url_elem = ET.SubElement(urlset, "url")
            loc = ET.SubElement(url_elem, "loc")
            loc.text = url
            lastmod = ET.SubElement(url_elem, "lastmod")
            lastmod.text = date.strftime("%Y-%m-%d")

        xml_content = minidom.parseString(ET.tostring(urlset)).toprettyxml(indent="  ")
        
        response = make_response(xml_content)
        response.headers["Content-Type"] = "application/xml"    

        return response
    except Exception as e:
        current_app.logger.error(f"Error generating sitemap: {str(e)}")
        return "An error occurred while generating the sitemap", 500

@main.route('/')
def home():
    parent_categories = TourParentCategory.query.all()
    # Fetch cars for all categories
    cars = Car.query.order_by(Car.category).all()
    # Fetch the latest 3 blog posts
    latest_blog_posts = Blog.query.order_by(Blog.created_at.desc()).limit(3).all()
    return render_template('home.html', 
                           parent_categories=parent_categories, 
                           cars=cars, 
                           VehicleCategory=VehicleCategory,
                           latest_blog_posts=latest_blog_posts)

@main.route('/car-rentals')
def car_rentals():
    parent_categories = TourParentCategory.query.all()
    cars = Car.query.order_by(Car.category, Car.name).all()
    categories = VehicleCategory.__members__.keys()
    return render_template('car_rentals.html', cars=cars, categories=categories, parent_categories=parent_categories)

@main.route('/tour-packages')
@main.route('/tour-packages/<string:category_slug>')
def tour_packages(category_slug=None):
    parent_categories = TourParentCategory.query.options(
        joinedload(TourParentCategory.subcategories).joinedload(TourSubCategory.tours)
    ).all()
    
    if category_slug:
        category = TourParentCategory.query.filter_by(slug=category_slug).first_or_404()
        tours_by_category = {category: []}
        for subcategory in category.subcategories:
            tours_by_category[category].extend(subcategory.tours)
    else:
        category = None
        tours_by_category = {}
        for parent_category in parent_categories:
            category_tours = []
            for subcategory in parent_category.subcategories:
                category_tours.extend(subcategory.tours)
            if category_tours:
                tours_by_category[parent_category] = category_tours

    return render_template('tour_packages.html', 
                           parent_categories=parent_categories,
                           category=category,
                           tours_by_category=tours_by_category,
                           TourType=TourType)

@main.route('/about')
def about():
    parent_categories = TourParentCategory.query.all()  
    return render_template('about.html', parent_categories=parent_categories)

@main.route('/contact', methods=['GET', 'POST'])
def contact():
    parent_categories = TourParentCategory.query.all()  
    form = ContactForm()
    if form.validate_on_submit():
        contact = Contact(
            name=form.name.data,
            email=form.email.data,
            subject=form.subject.data,
            message=form.message.data
        )
        db.session.add(contact)
        db.session.commit()
        flash('Your message has been sent. We will get back to you soon!', 'success')
        return redirect(url_for('main.contact'))
    return render_template('contact.html', form=form, parent_categories=parent_categories)

@main.route('/request-quote', methods=['GET', 'POST'])
def request_quote():
    parent_categories = TourParentCategory.query.all()  
    form = RequestQuoteForm()
    car_id = request.args.get('car_id')
    tour_id = request.args.get('tour_id')
    car = None
    tour = None
    request_type = None

    if car_id:
        car = Car.query.get_or_404(car_id)
        form.car_id.data = car_id
        form.car_name.data = car.name
        request_type = "car"
    elif tour_id:
        tour = Tour.query.get_or_404(tour_id)
        form.tour_id.data = tour_id
        form.tour_name.data = tour.title
        request_type = "tour"

    if form.validate_on_submit():
        try:
            # Convert empty strings to None for integer fields
            car_id = form.car_id.data
            car_id = int(car_id) if car_id and car_id.isdigit() else None
            
            tour_id = form.tour_id.data
            tour_id = int(tour_id) if tour_id and tour_id.isdigit() else None

            quote_request = QuoteRequest(
                first_name=form.first_name.data,
                last_name=form.last_name.data,
                num_adults=form.num_adults.data,
                num_children=form.num_children.data,
                date_from=form.date_from.data,
                date_to=form.date_to.data,
                country=form.country.data,
                email=form.email.data,
                phone=form.phone.data,
                message=form.message.data,
                car_id=car_id,
                tour_id=tour_id
            )
            
            db.session.add(quote_request)
            db.session.commit()

            # Prepare email message
            sender = ('Lanka Travel Bee', 'info@itint.site')
            msg = Message("New Quote Request",
                          sender=sender,
                          recipients=['info@lankatravelbee.com'])
            
            if form.car_name.data:
                request_type_str = f"Vehicle: {form.car_name.data}"
            elif form.tour_name.data:
                request_type_str = f"Tour: {form.tour_name.data}"
            else:
                request_type_str = "General Inquiry"

            msg.body = f"""
            New Quote Request:
            {request_type_str}
            Name: {form.first_name.data} {form.last_name.data}
            Adults: {form.num_adults.data}
            Children: {form.num_children.data}
            From: {form.date_from.data}
            To: {form.date_to.data}
            Country: {form.country.data}
            Email: {form.email.data}
            Phone: {form.phone.data}
            Message: {form.message.data}
            """
            mail.send(msg)

            flash('Thank you for your quote request. We will contact you shortly!', 'success')
            return redirect(url_for('main.thank_you'))
        
        except Exception as e:
            db.session.rollback()
            flash('An error occurred while processing your request. Please try again.', 'error')
            current_app.logger.error(f'Error in request_quote: {str(e)}')
            return render_template('request_quote.html', form=form, car=car, tour=tour, request_type=request_type, parent_categories=parent_categories)

    elif form.errors:
        for field, errors in form.errors.items():
            for error in errors:
                flash(f"{getattr(form, field).label.text}: {error}", 'error')

    return render_template('request_quote.html', form=form, car=car, tour=tour, request_type=request_type, parent_categories=parent_categories)

@main.route('/thank-you')
def thank_you():
    return render_template('thank_you.html')

@main.route('/cordelia-cruises')
def cordelia_cruises():
    return render_template('cordelia_cruises.html', popular_tours=get_popular_tours(), TourType=TourType)

@main.route('/tour/<string:slug>')
def tour_detail(slug):
    tour = Tour.query.options(
        joinedload(Tour.tour_days),
        joinedload(Tour.subcategory).joinedload(TourSubCategory.parent_category)
    ).filter_by(slug=slug).first_or_404()
    
    # Handle the case where tour.image might be None
    if tour.image is None:
        tour.image = 'default_tour_image.jpg'  # Set a default image
    
    # Update view count
    if tour.views is None:
        tour.views = 1
    else:
        tour.views += 1
    
    db.session.commit()

    # Fetch tour days if it's a multi-day tour
    tour_days = None
    if tour.type == TourType.MULTI_DAY_TOUR:
        tour_days = tour.tour_days

    return render_template('tour_detail.html', tour=tour, parent_categories=parent_categories, tour_days=tour_days)

@main.route('/blog')
def blog_list():
    parent_categories = TourParentCategory.query.all()  
    page = request.args.get('page', 1, type=int)
    blogs = Blog.query.order_by(Blog.created_at.desc()).paginate(page=page, per_page=10)
    return render_template('blog_list.html', blogs=blogs, parent_categories=parent_categories)

@main.route('/blog/<string:slug>')
def blog_detail(slug):
    parent_categories = TourParentCategory.query.all()  
    blog = Blog.query.filter_by(slug=slug).first_or_404()
    return render_template('blog_detail.html', blog=blog, parent_categories=parent_categories)

@main.route('/blog/category/<string:category>')
def blog_category(category):
    parent_categories = TourParentCategory.query.all()  
    page = request.args.get('page', 1, type=int)
    blogs = Blog.query.filter_by(category=category).order_by(Blog.created_at.desc()).paginate(page=page, per_page=10)
    return render_template('blog_category.html', category=category, blogs=blogs, parent_categories=parent_categories)

@main.route('/terms-and-conditions')
def terms_and_conditions():
    return render_template('terms_and_conditions.html')

@main.route('/privacy-policy')
def privacy_policy():
    return render_template('privacy_policy.html')

@main.route('/sitemap')
def sitemap_html():
    return render_template('sitemap.html')