from flask_wtf import FlaskForm
from flask_wtf.file import FileField, FileAllowed
from wtforms import StringField, TextAreaField, PasswordField, IntegerField, FloatField, SubmitField, BooleanField, SelectField, FieldList, FormField
from wtforms.validators import DataRequired, NumberRange, Optional, Length
from app.models import VehicleCategory, TourParentCategory, TourSubCategory

class LoginForm(FlaskForm):
    username = StringField('Username', validators=[DataRequired()])
    password = PasswordField('Password', validators=[DataRequired()])
    submit = SubmitField('Log In')

class CarForm(FlaskForm):
    name = StringField('Car Name', validators=[DataRequired()])
    capacity = StringField('Capacity', validators=[DataRequired()])
    price_per_day = FloatField('Price per Day', validators=[DataRequired(), NumberRange(min=0)])
    image = FileField('Image', validators=[FileAllowed(['jpg', 'png', 'jpeg', 'avif','webp'], 'Images only!')])
    category = SelectField('Category', choices=[(cat.name, cat.value) for cat in VehicleCategory], validators=[DataRequired()])
    has_ac = BooleanField('Air Conditioning')
    has_usb = BooleanField('USB Ports')
    has_player = BooleanField('Music Player')
    has_bluetooth = BooleanField('Bluetooth')
    luggage_space = StringField('Luggage Space')
    submit = SubmitField('Add Car')

class TourDayForm(FlaskForm):
    title = StringField('Day Title', validators=[DataRequired()])
    description = TextAreaField('Day Description', validators=[Optional()])

class TourForm(FlaskForm):
    title = StringField('Tour Title', validators=[DataRequired()])
    subtitle = StringField('Tour Subtitle', validators=[Optional()])
    description = TextAreaField('Description', validators=[Optional()])
    itinerary = TextAreaField('Itinerary', validators=[Optional()])
    inclusions = TextAreaField('Inclusions', validators=[Optional()])
    exclusions = TextAreaField('Exclusions', validators=[Optional()])
    additional_info = TextAreaField('Additional Information', validators=[Optional()])
    image = FileField('Tour Image', validators=[FileAllowed(['jpg', 'png', 'jpeg', 'avif','webp'], 'Images only!')])
    parent_category = SelectField('Tour Parent Category', coerce=int, validators=[DataRequired()])
    subcategory = SelectField('Tour Subcategory', coerce=int, validators=[DataRequired()])
    tour_days = FieldList(FormField(TourDayForm), min_entries=1)
    submit = SubmitField('Add Tour')

    def __init__(self, *args, **kwargs):
        super(TourForm, self).__init__(*args, **kwargs)
        self.parent_category.choices = [(c.id, c.name) for c in TourParentCategory.query.order_by('name')]
        self.subcategory.choices = [(c.id, f"{c.parent_category.name} - {c.name}") for c in TourSubCategory.query.join(TourParentCategory).order_by(TourParentCategory.name, TourSubCategory.name)]
        if len(self.tour_days) == 0:
            self.tour_days.append_entry()

class TourParentCategoryForm(FlaskForm):
    name = StringField('Category Name', validators=[DataRequired(), Length(max=100)])
    submit = SubmitField('Add Parent Category')

class TourSubCategoryForm(FlaskForm):
    name = StringField('Subcategory Name', validators=[DataRequired(), Length(max=100)])
    parent_category = SelectField('Parent Category', coerce=int, validators=[DataRequired()])
    image = FileField('Image', validators=[
        FileAllowed(['jpg', 'png', 'jpeg', 'avif', 'webp'], 'Images only! Allowed formats: JPG, PNG, JPEG, AVIF, WebP')
    ])
    submit = SubmitField('Add Subcategory')

    def __init__(self, *args, **kwargs):
        super(TourSubCategoryForm, self).__init__(*args, **kwargs)
        self.parent_category.choices = [(c.id, c.name) for c in TourParentCategory.query.order_by('name')]

class BlogForm(FlaskForm):
    title = StringField('Title', validators=[DataRequired(), Length(max=200)])
    content = TextAreaField('Description', validators=[DataRequired()])
    feature_image = FileField('Feature Image', validators=[FileAllowed(['jpg', 'png', 'jpeg', 'gif', 'avif', 'webp'], 'Images only!')])
    gallery_image_1 = FileField('Gallery Image 1', validators=[FileAllowed(['jpg', 'png', 'jpeg', 'gif', 'avif', 'webp'], 'Images only!')])
    gallery_image_2 = FileField('Gallery Image 2', validators=[FileAllowed(['jpg', 'png', 'jpeg', 'gif', 'avif', 'webp'], 'Images only!')])
    gallery_image_3 = FileField('Gallery Image 3', validators=[FileAllowed(['jpg', 'png', 'jpeg', 'gif', 'avif', 'webp'], 'Images only!')])
    meta_description = StringField('Meta Description', validators=[Length(max=160)])
    meta_keywords = StringField('Meta Keywords', validators=[Length(max=200)])
    submit = SubmitField('Submit')